from ctypes import *
import rhinoscriptsyntax as rs
import Rhino.Geometry as rg

def AlphaShape2D(generatrixList, alpha, regularized = False):
    """
    | AlphaShape2D(generatrixList, alpha, regularized = False) | generate 2d Alpha shape
    |               what: this is wrapping CGAL function (https://www.cgal.org/)
    |                       see more about alpha shape http://doc.cgal.org/latest/Alpha_shapes_2/index.html
    |               Paraemeters:
    |                   generatrixList = list of Rhino.Geometry.Point3d 
    |                                       if Point3d z coordinate is not 0. the result would be at the average of z
    |                   alpha = float number for alpha parameter, too small, too big number does not generate anything
    |                   
    |                   regularized = True / False, switch to regularize the alpha shape
    |               Returns:
    |                   list of new curve ids
    |
    """
    dll = cdll.cgal_wrap

    generatrixarray = []
    for pt in generatrixList:
        generatrixarray.append(pt.X)
        generatrixarray.append(pt.Y)
        generatrixarray.append(pt.Z)
        
    size = len(generatrixarray)
    #ctype conversion
    c_gene_array = (c_double * size)(*generatrixarray)
    c_len_gene = c_int(len(generatrixList))
    c_alpha = c_double(alpha)
    c_regularized = c_bool(regularized)
    dll.gen2DAlphaShape.restype = type(POINTER(c_double)())    
    res = dll.gen2DAlphaShape(c_gene_array,c_len_gene,c_alpha,c_regularized)

    segs = []    
    if res:
        psets = int(res[0] / 6) 
        for i in range(psets):
            x0 = res[i*6 + 1]
            y0 = res[i*6 + 2]
            z0 = res[i*6 + 3]
            x1 = res[i*6 + 4]
            y1 = res[i*6 + 5]
            z1 = res[i*6 + 6]
            seg = rs.AddLine([x0,y0,z0],[x1,y1,z1])
            segs.append(seg)               
            
    return segs


def VoronoiWeighted2DDiagram(generatrixList, heightAt = 0., rect = None):
    """
    | VoronoiWeighted2DDiagram(generatrixList, heightAt = 0., rect = None) | generate 2d weighted voronoi diagram shape
    |               what: this is wrapping CGAL function (https://www.cgal.org/)
    |                       see more about weighted voronoi
    |                       https://doc.cgal.org/4.2/CGAL.CGAL.2D-Apollonius-Graphs--Delaunay-Graphs-of-Disks-/html/index.html
    |               Paraemeters:
    |                   generatrixList = list of Rhino.Geometry.Point3d 
    |                                    Point3d.X for x coordinate
    |                                    Point3d.Y for y coordinate
    |                                    Point3d.Z for weight
    |                   heightAt = float number, z coordinate for generated diagram
    |                   
    |                   rect = closed rectangle curve for voronoi diagram triming
    |               Returns:
    |                   nested list of new curve ids for each cells
    |
    """
    dll = cdll.cgal_wrap

    generatrixarray = []
    for pt in generatrixList:
        generatrixarray.append(pt.X)
        generatrixarray.append(pt.Y)
        generatrixarray.append(pt.Z)
        coord = rg.Point3d(pt.X,pt.Y,heightAt)
    
    minx = miny = maxx = maxy = 0.0
    if rect != None:
        bbox = rs.BoundingBox(rect)
        if bbox != None:
            minx = bbox[0].X
            miny = bbox[0].Y
            maxx = bbox[2].X
            maxy = bbox[2].Y
            
    size = len(generatrixarray)
    #ctype conversion
    c_gene_array = (c_double * size)(*generatrixarray)
    c_len_gene = c_int(len(generatrixList))
    c_heightAt = c_double(heightAt)
    c_minx = c_double(minx)
    c_miny = c_double(miny)
    c_maxx = c_double(maxx)
    c_maxy = c_double(maxy)
    dll.gen2DVoronoiDiagramC.restype = type(POINTER(c_double)())
    res = dll.gen2DVoronoiDiagramC(c_gene_array,c_len_gene,c_heightAt,c_minx,c_miny,c_maxx,c_maxy)

    cells = []
    if res:
        num_cell = int(res[0])
        for i in range(1, num_cell + 1):
            num_edge = int(res[i])
            cells.append(num_edge)

        index = 1 + len(cells)
        edge_pt_list = []
        for c in cells:
            ptlist = []
            for i in range(c):
                num_pt = int(res[index])
                ptlist.append(num_pt)
                index += 1
            edge_pt_list.append(ptlist)

        for cell in edge_pt_list:
            edge_ids = []
            for edge in cell:
                ptlist = []
                for i in range(edge):
                    x = res[index]
                    index += 1
                    z = res[index]
                    index += 1
                    y = res[index]
                    index += 1
                    if x != x or y != y:
                        pass
                    else:
                        pt = rg.Point3d(x,y,z)
                        ptlist.append(pt)
                edge_ids.append(rs.AddPolyline(ptlist))
            cells.append(edge_ids)
    #dll.delarray(res)
    return cells